/*****************************************************************************
*                           Freescale Semiconductor                          *
*                                                                            *
*  Project    : AN3815 - MC56F8006 Modular Pixel Matrix                      *
*  Version    : 0.1                                                          *
*  Date       : 16/Apr/2009                                                  *
*  Authors    : Alexandre Dias, Bruno Bastos                                 *
*               Humberto Carvalho, Leonardo Mangiapelo                       *
*               Renato Frias                                                 *
*                                                                            *
* -------------------------------------------------------------------------- *
* THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR        *
* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES  *
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.    *
* IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,  *
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES         *
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR         *
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)         *
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,        * 
* STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING      *
* IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF             *
* THE POSSIBILITY OF SUCH DAMAGE.                                            *
* -------------------------------------------------------------------------- *
*  Description:                                                              *
* This software complements AN3815 - Implementing a Modular High Brightness  *
* RGB LED Network. The aim of the code is to proof the concept described on  *
* the application note document. It also serves as a starting point for      * 
* applications with MC56F8006 and some of its peripherals                    *
* -------------------------------------------------------------------------- *
*                                                                            *
*  File: sci.c	                                                             *
*                                                                            *
*     Interfaces with SCI registers on 56F8006. 							 * 
*                                                                            *
******************************************************************************/

/************************* Include Files *************************************/

#include "util.h"
#include "sci.h"
#include "MC56f8006.h"

/********************************* Type Definition ***************************/

typedef unsigned char byte;

/******************************** Prototypes *********************************/

//! RX data buffer
static uint8_t sci_buffer[SCI_RX_BUFFER_SIZE];
//! Buffer pointer for data In.
static uint8_t sci_ptr_in;
//! Buffer pointer for data Out.
static uint8_t sci_ptr_out;

/*********************** Function's Implementation ***************************/

//! Initializes Sci.
/*! 
	\param baudrate value for SCI.
	
	Set all SCI related functionality.
*/
void sci_init(uint32_t baudrate) 
{
	uint16_t	sbr;
	uint8_t	frac_sbr;
	
	/* Configure GPIOB6 and GPIOB7 as RXD and TXD */
	SIM_GPSB1 = 0;
	GPIOB_PER |= (GPIOB_PER_PER_6 | GPIOB_PER_PER_7);
	
	/* Enable clock to SCI */
	SIM_PCE |= SIM_PCE_SCI;
	
	sbr = (SCI_CLOCK_HZ>>4)/ baudrate;
	frac_sbr = ((SCI_CLOCK_HZ>>1)/baudrate - (sbr<<3)) & 0x7;
	SCI_RATE = (sbr<<3) | frac_sbr;
	
	//SCI_CTRL1 = (SCI_CTRL1_TE | SCI_CTRL1_RE);// without interrupt
	SCI_CTRL1 = (SCI_CTRL1_TE | SCI_CTRL1_RE | SCI_CTRL1_RFIE);// with interrupt
		
}

//! Returns if a byte was received by the SCI.
/*! 
	\return 1 for byte received, 0 otherwise.
	
	Tests buffer out and in pointers to check if data was received.
	
*/

uint8_t sci_byte_received(void)
{
	if (sci_ptr_out != sci_ptr_in)
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

//! Returns a byte was received from the SCI.
/*! 
	\return the next byte on the rx buffer. 
	
	Gets data from the RX Buffer and updates output pointer.
	Returns "-1" if no data was received. (A better error treatment should be implemented.)
	
*/
int8_t sci_read_byte(void)
{
	int8_t data;

	if (sci_ptr_out != sci_ptr_in)
	{
		data = sci_buffer[sci_ptr_out++]; 
		if(sci_ptr_out == SCI_RX_BUFFER_SIZE)
		{
			sci_ptr_out =0;
		}
		return data;
	}
	else
	{
		// No data available
		return -1;
	}
		
}

//! SCI RX Interrupt, it puts rx data on the buffer.
/*! 
	Stores rx data on the buffer and updates pointers.	
*/
#pragma interrupt on
#pragma section INTERRUPT_SERVICE_ROUTINES begin
void sci_isr(void)
{
	int8_t dummy;
  	
  	if (SCI_STAT & SCI_STAT_RDRF)
  	{
  		dummy = SCI_STAT; 
		sci_buffer[sci_ptr_in++] = SCI_DATA;
		
		if(sci_ptr_in == SCI_RX_BUFFER_SIZE)
		{
			sci_ptr_in = 0;
		}
		//if there is buffer overflow move out pointer
		if(sci_ptr_in == sci_ptr_out)
		{
			sci_ptr_out++;
			if(sci_ptr_out == SCI_RX_BUFFER_SIZE)
			{
				sci_ptr_out = 0;
			}
		}
  	}  	
	return;	
}
#pragma section INTERRUPT_SERVICE_ROUTINES end
